from osi3.osi_sensordata_pb2 import SensorData
import struct
import math


class OsiRadarSensor:
    """Wrapper for OSI Python interface"""

    def __init__(self, sensor_id=1, timestamp_sec=0, timestamp_nanos=0):
        self.sensordata = SensorData()
        self.sensordata.sensor_id.value = sensor_id
        self.sensordata.timestamp.seconds = timestamp_sec
        self.sensordata.timestamp.nanos = timestamp_nanos
        self.radar_sensor = self.sensordata.feature_data.radar_sensor.add()

    def add_detection(self, distance, azimuth, radial_velocity, rcs):
        detection = self.radar_sensor.detection.add()
        detection.position.distance = distance
        detection.position.azimuth = azimuth
        detection.radial_velocity = radial_velocity
        detection.rcs = rcs

    def serialize_to_byte_msg(self):
        bytes_buffer = self.sensordata.SerializeToString()
        return bytes_buffer


def generate_osi_msg(ts_sec, ts_nanos, range, azimuth, velocity, rcs):
    """
    Generates OSI message for the specified target.

    Returns:
        buffer containing serialized OSI message
    """
    rad = OsiRadarSensor(1, ts_sec, ts_nanos)
    rad.add_detection(range, azimuth, velocity, rcs)
    msg = rad.serialize_to_byte_msg()
    return msg


def range_sweep_const_power(interval, range_start, range_stop, vel, attenuation, az, f_center, airgap=0.0):
    """
    Yields OSI messages for an object moving towards radar at constant velocity

    Parameters:
        interval: interval at which this function is called in seconds
        range_start: start of range sweep in m
        range_stop: stop of range sweep in m
        vel: radial velocity of target in m/s
        attenuation: attenuation of input signal by the AREG800A in dB
        az: azimuth of target in radians (positive turns counter-clockwise)
        airgap: distance between frontend and radar sensor in meters. Set to zero for conducted setup.

    Returns:
        serialized OSI message
    """
    if range_start < range_stop:
        raise ValueError("range_start must be greater than range_stop")
    if vel > 0.0:
        raise ValueError("Radial velocity must be negative for a target moving towards the radar")
    if interval < 0.01:
        raise ValueError("Update interval for scenarios should be >= 10ms")

    c0 = 299700000
    timestamp_sec = 0
    timestamp_nanos = 0
    range_step = vel * interval
    curr_range = range_start
    while curr_range >= range_stop:
        # calculate RCS for each timestamp to reach the specified signal attenuation
        # -> signal attenuation constant over range
        curr_rcs = ((c0 / f_center) ** 2) * (curr_range ** 4) / (4 * math.pi) / (airgap ** 4) / (10 ** (attenuation/10))
        # convert RCS from m2 into dBm2
        curr_rcs_dbsm = 10 * math.log10(curr_rcs)
        msg = generate_osi_msg(timestamp_sec, timestamp_nanos, curr_range, az, vel, curr_rcs_dbsm)
        timestamp_nanos += int(interval * 1e9)
        curr_range += range_step
        if timestamp_nanos > 1000000000:
                timestamp_sec += 1
                timestamp_nanos = 0
        yield msg


if __name__ == "__main__":
    # scenario settings
    update_interval = 0.1   # scenario update rate in seconds
    start = 120.0           # m
    stop = 20.0             # m
    velocity = -10.0        # m/s
    signal_attenuation = 30 # dB
    azimuth = 0.0           # rad
    airgap = 0.5            # m
    f_center = 76.5e9       # Hz

    buf = bytes()
    osi_msg = range_sweep_const_power(update_interval, start, stop, velocity, signal_attenuation, azimuth, f_center, airgap)
    while True:
        try:
            next_msg = next(osi_msg)
        except StopIteration:
            print('End of scenario')
            break
        buf += struct.pack("<L", len(next_msg)) + next_msg
        
    f = open("range_sweep_const_power.osi", "wb")
    f.write(buf)
    f.close()
